/*

  
	io.cpp 
	Author: Stephen Pendleton
	11/16/97
	
	  This file contains most of the input/output for the client. This includes all the
	  parsing procedures and user input handlers.
*/

#include <windows.h>
#include <windowsx.h>
#include <Commctrl.h>
#include <Winsock.h>
#include <string.h>
#include <malloc.h>
#include <winnls.h>

#ifndef _WIN32_WCE
#include <stdio.h>
#endif

#include "resource.h"
#include "IrcCE.h"

extern int CLIENT_WIDTH;
extern int CLIENT_HEIGHT;
extern int CHANNEL_WINDOW_WIDTH;
extern int CHANNEL_WINDOW_HEIGHT;
extern HANDLE hEditWnd, hWndBox, hWndInfoWin;
extern PROC lpfnOldEditProc;
extern void PrintToScreen(char * buffer, char *dest);
extern void AddNameToInfoWin(const char *name);
extern void RemoveNameFromInfoWin(const char *name);
extern void ClearNamesFromInfoWin();
extern void AddNamesToInfoWin(const char *);
extern void ChangeNameInInfoWin(char *name, char *newname);
extern void FillInfoWin(char *channel);
extern DWORD WINAPI InitiateDCCCHATThreadHandler(LPVOID remote_user);
extern DWORD WINAPI InitiateDCCSENDThreadHandler(LPVOID thread_param);
extern DWORD WINAPI AcceptDCCCHATThreadHandler(LPVOID remote_user);
extern DWORD WINAPI AcceptDCCSENDThreadHandler(LPVOID remote_user);

extern SOCKET csock;
extern SOCKET ident_socket;
extern char Nickname[30],Userinfo[50];
extern char CurChannel[100];
extern HANDLE hMainWnd;
extern BOOL nickname_taken;

struct Channel * pChannelList;


enum actions {UNKNOWN,JOIN,PART,QUIT,KICK,TOPIC,PRIVMSG,CMSG,QUERY,CTCP,CTCP_DCC,NICK};

#ifdef _WIN32_WCE
void HandleUserInput(TCHAR *inp);
#else
void HandleUserInput(char *inp);
#endif

void WriteOutToSocket(SOCKET sock, char * buf);
char * ParseOutString(char * line, int *action);
char * ParseInString(char *, char *);
void AddNewLine(struct Channel *);
char * GetToken(const char * buffer, int *index, const char *seps);
char * ParseIrc(char *, char *);
void ParseDCC(char *to, char *param);
size_t __cdecl strspn (const char * string,const char * control);
int __cdecl strcmpi(const char * dst, const char * src);
void ActMode(char *channel, char *modes, char *to);
void AddMode(char *channel,char *to,char mode);
void RemoveMode(char *channel,char *to,char mode);
void ActJoin(char * nick, char *channel);
void ActPart(char * nick, char *channel);
void ActNick(char * nick, char *to);
void ActKick(char * nick, char *channel);
void ActQuit(char * nick, char *trailing);
struct Channel * GetChannel(char *);
void SetupNewChannel(char *channel, BOOL select_channel, int window_type, int socket);
Channel * CloseChannel(char *name);
Channel * CreateChannel(char *, int window_type, int sock);
BOOL ChannelExists(char *);
void AddNameToChannel(char *channel, char *name);
void AddNamesToChannel(char *channel, char *name);
void RemoveNameFromChannel(char *channel, char *name);
void RemoveNameFromAllChannels(char *name);
void RemoveNamesFromChannel(struct Channel *chan);
struct Names * FindNameInChannel(struct Channel *chan, char *name);
void SelectChannelInChannelBox(char *channel);
void RemoveChannelFromChannelBox(char *channel);
char * ParseCTCP(char *com, char *line, char * nick);
char * ActCTCP(char * command, char *nick, char *trailing);
Channel * DeleteChannel(char * name);
void MakeChannelCurrent( char * channel);
void ShutdownSockets();
HANDLE ScreenBufferEvent;
unsigned int ScrollPosition;




/****************************************************************
This function handles all the typed user input.
*****************************************************************/
#ifdef _WIN32_WCE
void HandleUserInput(TCHAR *inp)
{
	TCHAR temp[30],outs[200];
#else
void HandleUserInput(char *inp)
{
#endif
	
	char binp[200];
	char * pParsed;
	int action;
	struct Channel * chan;

	
#ifdef _WIN32_WCE
	wcstombs( binp, inp, 199);
#else
	strcpy(binp,inp);
#endif
	
	chan = GetChannel(CurChannel);
	
	if (chan->window_type == WINDOW_TYPE_CHANNEL)
	{
		pParsed=ParseOutString(binp,&action);
		
		switch(action)
		{
		case PRIVMSG:
			if (csock!=INVALID_SOCKET)
			{
				
#ifdef _WIN32_WCE
				mbstowcs( temp,  Nickname, 25 );
				wsprintf(outs,L"<%s> %s\n", temp, inp);
				wcstombs( binp, outs, 199 );
#else
				sprintf(binp,"<%s> %s\n", Nickname, inp);
#endif
				strcat(pParsed,"\r\n");
				WriteOutToSocket(csock, pParsed);
				PrintToScreen(binp,CurChannel);
			}
			break;
			
		case QUERY:
			if (csock!=INVALID_SOCKET)
			{
				if (pParsed!="")
					SetupNewChannel(pParsed, FALSE, WINDOW_TYPE_CHANNEL, csock);
			}
			break;
			
		case CTCP_DCC:
			if (csock!=INVALID_SOCKET)
			{
				if (pParsed!="")
					PrintToScreen("Attempting DCC...\n", "status");
			}
			break;
			
		default:
			strcat(pParsed,"\r\n");
			WriteOutToSocket(csock, pParsed);
			break;
		}
		LocalFree(pParsed);
	}
	else if (chan->window_type == WINDOW_TYPE_DCC_CHAT)
	{
		strcat(binp,"\n");
		WriteOutToSocket(chan->sock, binp);
		
#ifdef _WIN32_WCE
		mbstowcs( temp,  Nickname,  25 );
		wsprintf(outs,L"<%s> %s\n", temp, inp);
		wcstombs( binp, outs, 220 );
#else
		sprintf(binp,"<%s> %s\n", Nickname, inp);
#endif
		PrintToScreen(binp,CurChannel);
	}
	
	
	LocalFree(inp);
	
}

/****************************************************************
Utility function that writes data to a socket
*****************************************************************/

void WriteOutToSocket(SOCKET sock, char * buf)
{
	if (sock!=INVALID_SOCKET)
		send(sock,buf,strlen(buf),0);
}

/****************************************************************
Window procedure for the edit window.
*****************************************************************/

LRESULT CALLBACK EditProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM uParam)
{
	
	BYTE nVirtKey;

    switch (message) {
		
		
	case WM_KEYDOWN:
		nVirtKey = (BYTE) (wParam&0xFF);

		if (nVirtKey==VK_RETURN)
		{
			#ifdef _WIN32_WCE
			TCHAR * line;
			line = (TCHAR *)LocalAlloc(LPTR, 199);
			GetWindowText(hEditWnd, line, 199);
			#else 
			char * line;
			line = (char *)LocalAlloc(LPTR, 199);
			GetWindowText(hEditWnd, line, 199);
			#endif
			HandleUserInput(line);
#ifdef _WIN32_WCE
			SetWindowText(hEditWnd,L"");
#else
			SetWindowText(hEditWnd,"");
#endif
			return 0;
		}
		break;	
	}

	return CallWindowProc(lpfnOldEditProc, hEditWnd, message, wParam, uParam);
	
}


/****************************************************************
Parses the text the user types in and decides what action to take.
*****************************************************************/

char * ParseOutString(char * line, int *action)
{
	char *buff;
	int indexb;
	char *to,*command, *param;
	
	indexb = 0;
	*action = -1;
	to=NULL;
	command=NULL;
	
	buff = (char *)LocalAlloc(LMEM_FIXED, 250);
	
	
	if (line[0] == '/')
	{
		indexb++;
		command=GetToken(line,&indexb," \n");
		
		if (strcmpi(command,"QUIT")==0)
		{	
			strcpy(buff,"QUIT :");
			to = GetToken(line,&indexb,"\n");
			strcat(buff,to);
			LocalFree(to);
			*action =QUIT;
		}
		
		else if (strcmpi(command,"QUERY")==0)
		{	
			to = GetToken(line,&indexb," \n");
			strcpy(buff, to);
			LocalFree(to);
			*action =QUERY;
			
		}
		else if (strcmpi(command,"CTCP")==0)
		{	
			strcpy(buff,"PRIVMSG ");
			to = GetToken(line,&indexb," ");
			strcat(buff,to);
			LocalFree(to);
			strcat(buff," :\001");
			to = GetToken(line,&indexb," \n");
			strcat(buff,to);
			strcat(buff,"\001");
			*action =CTCP;
		}
		
		else if (strcmpi(command,"TOPIC")==0)
		{	
			strcpy(buff,"TOPIC ");
			to = GetToken(line,&indexb," ");
			strcat(buff,to);
			LocalFree(to);
			strcat(buff," :");
			to = GetToken(line,&indexb,"\n");
			strcat(buff,to);
			LocalFree(to);
			*action =TOPIC;
		}
		else if (strcmpi(command,"ME")==0)
		{		
			
			to = GetToken(line,&indexb,"\n");
			strcpy(buff, "***");
			
			strcat(buff, Nickname);
			strcat(buff, " ");
			strcat(buff, to);
			strcat(buff, "\n");
			PrintToScreen(buff, CurChannel);
			
			strcpy(buff,"PRIVMSG ");
			strcat(buff,CurChannel);
			strcat(buff," :\001ACTION ");
			
			strcat(buff,to);
			strcat(buff,"\001");
			LocalFree(to);
			*action =CTCP;
		}
		else if (strcmpi(command,"MSG")==0)
		{		
			strcpy(buff,"PRIVMSG ");
			to = GetToken(line,&indexb," \n");
			strcat(buff,to);
			LocalFree(to);
			strcat(buff," :");
			to = GetToken(line,&indexb,"\n");
			strcat(buff,to);
			*action =CMSG;
		}
		else if (strcmpi(command,"DCC")==0)
		{	
			strcpy(buff,"DCC ");
			to = GetToken(line,&indexb," \n");
			strcat(buff,to);
			param = GetToken(line,&indexb,"\n");
			strcat(buff," ");
			strcat(buff, to);
			ParseDCC(param, to);
			LocalFree(param);
			*action =CTCP_DCC;
		}
		/*
		else if (strcmpi(command,"LIST")==0)
		{	
		strcpy(buff,"LIST ");
		to = GetToken(line,&indexb,"\n");
		strcat(buff,to);
		*action =LIST;
		}
		*/
		else if (nickname_taken && strcmpi(command,"NICK")==0)
		{	
			nickname_taken = FALSE;
			to = GetToken(line,&indexb,"\n");
			strcpy(Nickname, to);
			LocalFree(to);
			strcpy(buff,line+1);
			*action =NICK;
		}
		else if (strcmpi(command,"KICK")==0)
		{	
			strcpy(buff,"KICK ");
			to = GetToken(line,&indexb," ");
			strcat(buff,to);
			LocalFree(to);
			to = GetToken(line,&indexb," \n");
			strcat(buff, " ");
			strcat(buff, to);
			LocalFree(to);
			to = GetToken(line,&indexb,"\n");
			strcat(buff," :");
			strcat(buff,to);
			LocalFree(to);
			*action =KICK;
		}
		else
		{
			strcpy(buff,line+1);
			*action =UNKNOWN;
		}
		
		if (command)
			LocalFree(command);
		if (to)
			LocalFree(to);
		}
		else
		{
			strcpy(buff,"PRIVMSG ");
			strcat(buff,CurChannel);
			strcat(buff," :");
			strcat(buff,line);
			*action =PRIVMSG;
		}
		
		
		return buff;
		
		
}

/****************************************************************
This function performs the login functions required by an irc 
server.
*****************************************************************/

void Login()
{
	
	char buff[100];
	strcpy(buff,"USER ");
	strcat(buff,Nickname);
	strcat(buff," server1 server2 :");
	strcat(buff,Userinfo);
	strcat(buff,"\r\n");
	WriteOutToSocket(csock, buff);
	
	strcpy(buff,"NICK ");
	strcat(buff,Nickname);
	strcat(buff,"\r\n");
	
	WriteOutToSocket(csock, buff);
	
}


/****************************************************************
This function gets a token from an indexed buffer based on the
seperators passed.
*****************************************************************/

char * GetToken(const char * buffer, int *index, const char *seps)
{
	int indexb;
	char *retbuf = NULL;
	
	// skip spaces
	indexb = strspn(buffer+*index," ");
	
	*index +=indexb;
	
	indexb = strspn(buffer+*index,seps);
	
	*index +=indexb;
	
	indexb = strcspn(buffer+*index,seps);
	
	if (indexb)
	{
		retbuf = (char *)LocalAlloc(LMEM_FIXED, 1000);
		strncpy(retbuf,buffer+*index,indexb);
		*index +=indexb;
		retbuf[indexb]=0;
	}
	
	
	
	
	return retbuf;
	
}

/****************************************************************
Adds a line of text to a channels message list. If the destination
is the current channel then the client window is redrawn, displaying
the new text.
*****************************************************************/

void PrintToScreen(char * buffer, char * dest)
{
	
	char * pCR;
	char *buff;
	Channel * chan;
	RECT clientrect;
	
	if (WaitForSingleObject( ScreenBufferEvent,2000)!=WAIT_OBJECT_0)
		return; 
	
	buff = (char *)LocalAlloc(LMEM_FIXED, 1000);

	chan = GetChannel(dest);
	
	
	if (chan->pMsgList==NULL)
		AddNewLine(chan);
	
	do
	{
		
		if (chan->pEndMsgList->state==2)
			AddNewLine(chan);
		
		
		pCR=strchr(buffer,'\n');
		
		if (pCR)
		{
			strncpy(buff,buffer,pCR-buffer);
			buff[pCR-buffer]=0;
			pCR++;
			buffer=pCR;
			
			strcat(chan->pEndMsgList->text, buff);
			chan->pEndMsgList->state=2;
		}
		else 
		{
			strcat(chan->pEndMsgList->text, buffer);
			chan->pEndMsgList->state=1;
		}
	} while (pCR && *pCR!=0);
	
	
	if (strcmpi(CurChannel,dest)==0)
	{
		clientrect.left = 0;
		clientrect.right = CHANNEL_WINDOW_WIDTH;
		clientrect.top = 0;
		clientrect.bottom = CHANNEL_WINDOW_HEIGHT;
		
		InvalidateRect(hMainWnd,&clientrect,TRUE); 
		UpdateWindow(hMainWnd);
		
	}
	
	LocalFree(buff);
	SetEvent(ScreenBufferEvent); 
	
}

/****************************************************************
Adds a new line to the message list of a channel
*****************************************************************/

void AddNewLine(Channel * chan)
{
	
	
	struct Msg * oldpMsgList;
	
	oldpMsgList=chan->pEndMsgList;
	
	chan->pEndMsgList=(Msg *)LocalAlloc(LMEM_FIXED,sizeof(Msg));
	chan->pEndMsgList->text=(char *)LocalAlloc(LMEM_FIXED,1000);
	
	*(chan->pEndMsgList->text)=0;
	chan->pEndMsgList->next=NULL;
	chan->pEndMsgList->prev=oldpMsgList;
	chan->pEndMsgList->state=0;
	
	if (oldpMsgList)
		oldpMsgList->next=chan->pEndMsgList;
	else
		chan->pMsgList=chan->pEndMsgList;
	
	if (chan->NumBufferLines>=NUM_SCREEN_LINES)
	{
		oldpMsgList=chan->pMsgList->next;
		LocalFree(chan->pMsgList->text);
		LocalFree(chan->pMsgList);
		chan->pMsgList=oldpMsgList;
		chan->pMsgList->prev=NULL;
		
	}
	
	else
		chan->NumBufferLines++;
	
}

/****************************************************************
Shutsdown the main socket
*****************************************************************/

void ShutdownSockets()
{
	if (ident_socket != INVALID_SOCKET)
	{
		shutdown(ident_socket,2);
		closesocket(ident_socket);
	}

	
	if (csock != INVALID_SOCKET)
	{
		shutdown(csock,2);
		closesocket(csock);
	}
}

/****************************************************************
Frees all the memory allocated for all the message lists of all
the channels
*****************************************************************/

void FreeScreenBuffer()
{
	
	struct Msg *pMsg;
	struct Channel * chan;
	struct Names * pNames;
	
	chan = pChannelList;
	
	while (chan)
	{
		
		pMsg = chan->pMsgList;
		
		while (pMsg)
		{
			pMsg=chan->pMsgList->next;
			LocalFree(chan->pMsgList->text);
			LocalFree(chan->pMsgList);
			chan->pMsgList=pMsg;
		}
		
		pNames=chan->pNamesList;
		while (pNames)		
		{	
			pNames=chan->pNamesList->next;
			LocalFree(chan->pNamesList);
			chan->pNamesList=pNames;	
		}
		
		LocalFree(chan->name);
		chan = pChannelList->next;
		LocalFree(pChannelList);
		pChannelList=chan;
	}
	
}

/****************************************************************
Intialized the message list of a channel
*****************************************************************/

void SetupScreenBuffer(Channel *chan)
{
	chan->pMsgList=NULL;
	chan->pEndMsgList=NULL;
	chan->NumBufferLines=0;
}

/****************************************************************
Creates a new channel
*****************************************************************/

Channel * CreateChannel(char * name, int window_type, int sock)
{
	
	Channel * chan;
	
	chan=(Channel *)LocalAlloc(LMEM_FIXED,sizeof(Channel));
	chan->name=(char *)LocalAlloc(LMEM_FIXED,100);
	chan->next=pChannelList;
	chan->pNamesList = NULL;
	chan->window_type = window_type;
	chan->sock = sock;
	chan->pMsgList = NULL;
	chan->pEndMsgList = NULL;
	pChannelList=chan;
	strcpy(chan->name,name);
	SetupScreenBuffer(chan);
	
	return chan;
	
}

/****************************************************************
Sends a part message to leave the current channel or if its a 
DCC channel it closes the window and deletes it.
*****************************************************************/

Channel * CloseChannel(char *name)
{
	
	char out[100];
	
	if (strcmpi(name, "status")==0)
		return NULL;
	
	if (name[0]=='#')
	{
		strcpy(out, "PART ");
		strcat(out, CurChannel);
		strcat(out, "\r\n");
		WriteOutToSocket(csock, out);
		return NULL;
	}
	else
		return DeleteChannel(name);
	
}

/****************************************************************
Closes and deletes all channels (except status)
*****************************************************************/

Channel * CloseAllChannels()
{
	Channel * chan, *prevchan, *nextchan;
	Msg * pMsg;
	struct Names * pNames;
	
	chan = pChannelList;
	prevchan = NULL;
	nextchan = pChannelList;
	
	while (chan)
	{
		if (strcmpi(chan->name,"status")!=0)
		{
			
			if (chan->window_type == WINDOW_TYPE_DCC_CHAT)
			{
				shutdown(chan->sock,2);
				closesocket(chan->sock);
			}
			
			pMsg = chan->pMsgList;
			
			while (pMsg)
			{
				pMsg=chan->pMsgList->next;
				LocalFree(chan->pMsgList->text);
				LocalFree(chan->pMsgList);
				chan->pMsgList=pMsg;
			}
			
			pNames = chan->pNamesList;
			
			while (pNames)
			{
				pNames=chan->pNamesList->next;
				LocalFree(chan->pNamesList);
				chan->pNamesList=pNames;
			}
			
			nextchan = chan->next;
			
			if (chan == pChannelList)
				pChannelList = chan->next;
			else
				prevchan->next = chan->next;
			
			RemoveChannelFromChannelBox(chan->name);
			LocalFree(chan->name);
			LocalFree(chan);
		}
		prevchan=chan;
		chan=chan->next;
	}
	
	
			
	MakeChannelCurrent("status");
	
	return nextchan;
	
}
/****************************************************************
Deletes a channel.
*****************************************************************/

Channel * DeleteChannel(char * name)
{
	
	Channel * chan, *prevchan, *nextchan;
	Msg * pMsg;
	struct Names * pNames;
	
	chan = pChannelList;
	prevchan = NULL;
	nextchan = pChannelList;
	
	while (chan)
	{
		if (strcmpi(chan->name,name)==0)
		{
			
			if (chan->window_type == WINDOW_TYPE_DCC_CHAT)
			{
				shutdown(chan->sock,2);
				closesocket(chan->sock);
			}
			
			pMsg = chan->pMsgList;
			
			while (pMsg)
			{
				pMsg=chan->pMsgList->next;
				LocalFree(chan->pMsgList->text);
				LocalFree(chan->pMsgList);
				chan->pMsgList=pMsg;
			}
			
			pNames = chan->pNamesList;
			
			while (pNames)
			{
				pNames=chan->pNamesList->next;
				LocalFree(chan->pNamesList);
				chan->pNamesList=pNames;
			}
			
			nextchan = chan->next;
			
			if (chan == pChannelList)
				pChannelList = chan->next;
			else
				prevchan->next = chan->next;
			
			
			LocalFree(chan->name);
			LocalFree(chan);
			
			RemoveChannelFromChannelBox(name);
			break;
		}
		prevchan=chan;
		chan=chan->next;
	}
	
	
	if (strcmpi(CurChannel, name)==0)
	{
		
		if (nextchan)
			MakeChannelCurrent(nextchan->name);
		else
			MakeChannelCurrent("status");
	}
	
	
	return nextchan;
}


/****************************************************************
Paints all the text on the main chat window.
*****************************************************************/

void PaintWindow (HDC hdc)
{
	
	HFONT hfnt, hOldFont;
	int textheight,lineheight;
	unsigned int count;
	RECT rect;
	struct Msg * pMsg;
	Channel * chan;
	
#ifdef _WIN32_WCE
	WCHAR mbstr[500];
#endif
	
	hfnt = GetStockObject(SYSTEM_FONT); 
	
	textheight = 0;
	
	if (hOldFont = SelectObject(hdc, hfnt)) 	
	{ 		
		SelectObject(hdc, hOldFont);     
		chan = GetChannel(CurChannel);
		pMsg = chan->pEndMsgList;
		
		count = 0;
		
		while (pMsg != NULL && ScrollPosition > count++)
			pMsg = pMsg->prev;
		
		while (pMsg != NULL && textheight < 220)
		{
			if (pMsg->state == 2)
			{
				
				rect.left = 0;
				rect.top = CHANNEL_WINDOW_HEIGHT - textheight;
				rect.right = CHANNEL_WINDOW_WIDTH;		
				rect.bottom = CHANNEL_WINDOW_HEIGHT - textheight;
				
#ifdef _WIN32_WCE
				mbstowcs( mbstr, pMsg->text, strlen(pMsg->text)+1);
				lineheight = DrawText(hdc, mbstr, -1, &rect, DT_NOCLIP |DT_NOPREFIX | DT_LEFT | DT_WORDBREAK | DT_CALCRECT );
				rect.top -= lineheight;
				textheight += DrawText(hdc, mbstr, -1, &rect, DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_WORDBREAK);
				
				//	ExtTextOut(hdc,0,22+y*14,0,NULL,mbstr,strlen(pMsg->text),NULL);
#else
				lineheight = DrawText(hdc, pMsg->text, -1, &rect, DT_NOCLIP |DT_NOPREFIX | DT_LEFT | DT_WORDBREAK | DT_CALCRECT );
				rect.top -= lineheight;
				textheight += DrawText(hdc, pMsg->text, -1, &rect, DT_NOCLIP |DT_NOPREFIX |DT_LEFT | DT_WORDBREAK);
#endif
			}
			pMsg=pMsg->prev;
		}
	} 
	
}


// WinCE 1.0 doesnt have these functions.


/* Routine prototype */

char * __cdecl strrchr (const char * string,int ch)
{
        char *start = (char *)string;

        while (*string++)                       /* find end of string */
                ;
                                                /* search towards front */
        while (--string != start && *string != (char)ch)
                ;

        if (*string == (char)ch)                /* char found ? */
                return( (char *)string );

        return(NULL);
}

size_t strspn (const char * string,const char * control)
{
	const char *str = string;
	const char *ctrl = control;
	
	unsigned char map[32];
	int count;
	
	/* Clear out bit map */
	for (count=0; count<32; count++)
		map[count] = 0;
	
	/* Set bits in control map */
	while (*ctrl)
	{
		map[*ctrl >> 3] |= (1 << (*ctrl & 7));
		ctrl++;
	}
	
	
	/* 1st char NOT in control map stops search */
	if (*str)
	{
		count=0;
		while (map[*str >> 3] & (1 << (*str & 7)))
		{
			count++;
			str++;
		}
		return(count);
	}
	return(0);
	
	
}

int __cdecl strcmpi(const char * dst, const char * src)
{
	int f,l;
	do {
		if ( ((f = (unsigned char)(*(dst++))) >= 'A') && (f <= 'Z') )
			f -= ('A' - 'a');
		
		if ( ((l = (unsigned char)(*(src++))) >= 'A') && (l <= 'Z') )
			l -= ('A' - 'a');
	} while ( f && (f == l) );
	return(f - l);
}


/****************************************************************
This function produces a client action based on the tokens 
passed in.
*****************************************************************/

char * Act(char * dest, char * source, char * command, char *middle, char *trailing)
{
	char * out=(char *)LocalAlloc(LMEM_FIXED,1000);
	char * ctcp_out;
	char * param1, *param2, *param3;
	int indexb;
	
	if (strcmpi(command,"JOIN")==0)
	{
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," has joined ");
		if (trailing)
			strcat(out,trailing);
		
		strcat(out,".*****");
		strcpy(dest,trailing);		
		ActJoin(source,trailing);
	}
	else if (strcmpi(command,"PART")==0)
	{
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," has left ");
		if (middle)
			strcat(out,middle);
		
		strcpy(dest,middle);
		strcat(out,".*****");
		ActPart(source,middle);
	}
	else if (strcmpi(command,"MODE")==0)
	{
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," sets mode ");
		indexb = 0;
		param1 = GetToken(middle,&indexb," ");
		param2 = GetToken(middle,&indexb," \r\n");
		param3 = GetToken(middle,&indexb,"\r\n");
	
		if (trailing)
		{
			strcat(out, trailing);
			ActMode("status", param1, trailing);
		}
		else if (param2 && param3)
		{
			strcat(out,param2);
			strcat(out," ");
			strcat(out,param3);
			ActMode(param1, param3, param2);
		}

		strcat(out,".*****");
		
		strcpy(dest,param1);
		LocalFree(param1);
		LocalFree(param2);
		if (param3)
			LocalFree(param3);
		
	}
	else if (strcmpi(command,"NICK")==0)
	{
		
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," is now known as ");
		if (trailing)
			strcat(out,trailing);
		strcat(out,".*****");
		
		ActNick(source,trailing);
	}
	else if (strcmpi(command,"TOPIC")==0)
	{
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," sets the topic of channel ");
		if (middle)
			strcat(out,middle);
		strcat(out," to ");
		if (trailing)
			strcat(out,trailing);
		strcat(out,".*****");
		strcpy(dest, middle);
	}
	else if (strcmpi(command,"INVITE")==0)
	{
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," invites ");
		if (middle)
			strcat(out,middle);
		strcat(out," to ");
		if (trailing)
			strcat(out,trailing);
		strcat(out,".*****");
	}
	else if (strcmpi(command,"KICK")==0)
	{
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," kicks ");
		indexb = 0;
		param1 = GetToken(middle,&indexb," ");
		param2 = GetToken(middle,&indexb," ");
		if (param2)
			strcat(out,param2);
		strcat(out," from ");
		if (param1)
			strcat(out,param1);
		strcat(out," (");
		if (trailing)
			strcat(out,trailing);
		strcat(out,") *****");
		strcpy(dest, param1);
		ActKick(param2, param1);
		LocalFree(param1);
		LocalFree(param2);
		ActPart(source,param1);
	}
	else if (strcmpi(command,"QUIT")==0)
	{
		
		strcpy(out,"*****");
		if (source)
			strcat(out,source);
		strcat(out," has quit");
		strcat(out," (");
		if (trailing)
			strcat(out,trailing);
		strcat(out,")");
		strcat(out,".*****");
		ActQuit(source, trailing);
	}
	else if (strcmpi(command,"PRIVMSG")==0 || strcmpi(command,"NOTICE")==0)
	{
		if (strcmp(source,"")==0)
			strcpy(source, "status");
		if (middle && middle[0]=='#')
			strcpy(dest, middle);
		else
		{
			if ( (trailing[0]!=1) && (ChannelExists(source)==FALSE) )
			{	
				SetupNewChannel(source, FALSE, WINDOW_TYPE_CHANNEL, csock);
				strcpy(out, "----> Private Message from: ");
				strcat(out, source);
				strcat(out," <----\n");
				PrintToScreen(out, CurChannel);
			}
			
			strcpy(dest, source);
		}
		
		if (trailing[0]==1)
		{
			ctcp_out = ParseCTCP(command, trailing, source);
			strcpy(out, ctcp_out);
			LocalFree(ctcp_out);
		//	strcpy(dest, "status");
		}
		else
		{
			strcpy(out,"<");
			if (source)
				strcat(out,source);
			strcat(out,"> ");
			if (trailing)
				strcat(out,trailing);
		}
	}
	else if ( strcmpi(command,"353")==0)
	{
		indexb = 0;
		param1 = GetToken(middle,&indexb," ");
		param2 = GetToken(middle,&indexb," ");
		param3 = GetToken(middle,&indexb, " ");
		
		strcpy(out,"Names:");
		if (trailing)
			strcpy(out,trailing);
		strcpy(dest, "status");
		AddNamesToChannel(param3,trailing);

		LocalFree(param1);
		LocalFree(param2);
		LocalFree(param3);
		
	}
	else if ( strcmpi(command,"433")==0)
	{
		nickname_taken = TRUE;
		strcpy(out,"Your nickname is already in use. Type /NICK <newnickname> to set a new nickname.");
		strcpy(dest, "status");
	}
	
	else if (strcmpi(command,"PING")==0)
	{
		strcpy(out,"PONG ");
		if (trailing)
			strcat(out,trailing);
		strcat(out,"\r\n");
		WriteOutToSocket(csock, out);
		
		strcpy(out,"*****You were pinged by ");
		if (trailing)
			strcat(out,trailing);
		strcat(out,".*****");
		
		
	}

	else 
	{
	/*	strcpy(out,"UM:");
		strcat(out,"F:");
		if (source)
			strcat(out,source);
		strcat(out,"C:");
		if (command)
			strcat(out,command);
		strcat(out,"M:");
		if (middle)
			strcat(out,middle);
		strcat(out,"T:");
		if (trailing)
			strcat(out,trailing);
			*/
		strcpy(out,">");
		if (source)
			strcpy(out,source);
		strcat(out," ");
		if (trailing)
			strcat(out,trailing);
		strcpy(dest, "status");
	}
	
	strcat(out,"\n");
	return out;	
}

/****************************************************************
This function parses a DCC request.
*****************************************************************/

void ParseDCC(char *to, char *param)
{
	unsigned long identthreadid;
	char *thread_param;
	
	thread_param = (char *)LocalAlloc(LMEM_FIXED, 100);
	
	strcpy(thread_param, to);
	
	if (strcmpi(param,"CHAT")==0)
		CreateThread(NULL,0,InitiateDCCCHATThreadHandler,thread_param,0,&identthreadid);
	else if (strcmpi(param, "SEND")==0)
		CreateThread(NULL,0,InitiateDCCSENDThreadHandler,thread_param,0,&identthreadid);
}

/****************************************************************
This function parses a CTCP request.
*****************************************************************/

char * ParseCTCP(char * com, char *line, char * nick)
{
	
	
	int indexb;
	char * command="", *trailing ="", *out="";
	
	indexb = 1;
	command = GetToken(line, &indexb, " \001");
	trailing = GetToken(line, &indexb, "\001\n");
	
	if (strcmpi(com,"PRIVMSG")==0)					// then the ctcp is being initiated by someone else
		out = ActCTCP(command, nick, trailing);
	
	else											// this is a reply from a ctcp we initiated.
	{
		out=(char *)LocalAlloc(LMEM_FIXED,1000);
		strcpy(out, "CTCP REPLY TO '");
		strcat(out, command);
		strcat(out, "' : ");
		strcat(out, trailing);
	}
	
	
	
	LocalFree(command);
	LocalFree(trailing);
	return out;
}

/****************************************************************
This function acts based upon a CTCP request.
*****************************************************************/

char * ActCTCP(char * command, char *nick, char *trailing)
{
	
	char * param, *param2;
	char * out=(char *)LocalAlloc(LMEM_FIXED,1000);
	
	int indexb;
	unsigned long threadid;
	
	
	if (strcmpi(command, "ACTION")==0)
	{
		
		strcpy(out, "***");
		if (nick)
			strcat(out, nick);
		if (trailing)
		{
			strcat(out," ");
			strcat(out, trailing);
		}
		
	}
	
	else if (strcmpi(command, "VERSION")==0)
	{
		
		
		strcpy(out, "NOTICE ");
		strcat(out, nick);
		strcat(out, " :\001VERSION IrcCE:");
		strcat(out, CLIENT_VERSION);
		strcat(out, ":WinCE\001");
		strcat(out,"\r\n");
		WriteOutToSocket(csock, out);
		
		strcpy(out, "***CTCP: ");
		strcat(out, nick);
		strcat(out, " requests your version.");
		
	}
	else if (strcmpi(command, "USERINFO")==0)
	{
		strcpy(out, "NOTICE ");
		strcat(out, nick);
		strcat(out, " :\001USERINFO :");
		strcat(out, Userinfo);
		strcat(out, "\001");
		strcat(out,"\r\n");
		WriteOutToSocket(csock, out);
		
		strcpy(out, "***CTCP: ");
		strcat(out, nick);
		strcat(out, " requests your userinfo.");
		
	}
	
	else if (strcmpi(command, "PING")==0)
	{
		strcpy(out, "NOTICE ");
		strcat(out, nick);
		strcat(out, " :\001PING :");
		strcat(out, trailing);
		strcat(out, "\001");
		strcat(out,"\r\n");
		WriteOutToSocket(csock, out);
		
		strcpy(out, "***CTCP: ");
		strcat(out, nick);
		strcat(out, " pinged you.");
		
	}
	else if (strcmpi(command, "DCC")==0)
	{
		indexb = 0;
		param2 = GetToken(trailing, &indexb, " ");
		
		if (strcmpi(param2, "CHAT")==0)
		{
			strcpy(out, "DCC CHAT REQUEST RECEIVED");
			param = (char *)LocalAlloc(LMEM_FIXED,200);
			strcpy(param, nick);
			strcat(param, " ");
			strcat(param, trailing);
			
			CreateThread(NULL,0,AcceptDCCCHATThreadHandler,param,0,&threadid);
		}
		else if (strcmpi(param2, "SEND")==0)
		{
			strcpy(out, "DCC SEND REQUEST RECEIVED");
			param = (char *)LocalAlloc(LMEM_FIXED,200);
			strcpy(param, nick);
			strcat(param, " ");
			strcat(param, trailing);
			
			CreateThread(NULL,0,AcceptDCCSENDThreadHandler,param,0,&threadid);
		}
		
		LocalFree(param2);
	}
	else
	{
		strcpy(out, "***CTCP: UNKNOWN CTCP '");
		if (command)
			strcat(out, command);
		strcat(out, "' FROM ");
		strcat(out, nick);
		
	}
	
	strcat(out,"\n");
	return out;
	
	}
	
	
	/****************************************************************
	This function parses a line read from the server.
	*****************************************************************/
	
	char * ParseIRC(char * line, char *dest)
	{
		
		char *to=NULL, *nick=NULL, *user=NULL, *host=NULL, *command=NULL, *trailing=NULL, *middle=NULL;
		char *out, *temp;
		char *source,*from;
		
		int indexb, dummy;
		
		
		indexb = 0;
				
		source = (char *)LocalAlloc(LMEM_FIXED, 300);		
		strcpy(source,"");
				
		if (line[0] == ':')
		{
			indexb = 1;
			if (strchr(line+indexb,' '))
			{
				dummy=0;				
				from = GetToken(line,&indexb," ");
				strcpy(source, from);
				if (strchr(from,'!'))
				{
					nick = GetToken(from,&dummy," !");
					strcpy(source, nick);
					if (nick)
						LocalFree(nick);
					user = GetToken(from,&dummy," @");

					if (user)
						LocalFree(user);
				}
				if (strchr(from,'@'))
				{
					host = GetToken(from+dummy,&dummy," ");
					if (host)
						LocalFree(host);
				}
				LocalFree(from);
				indexb++;				
			}			
			
		}
		
		if (strchr(line+indexb,' '))
		{
			command = GetToken(line,&indexb," ");
			indexb++;
		}
		
		
		if (strchr(line+indexb,':'))
		{
			middle=(char *)LocalAlloc(LMEM_FIXED,500);	
			strcpy(middle,"");
		
			do 
			{
				
				if (*(line+indexb)==':')
				{
					indexb++;
					trailing = GetToken(line,&indexb,"\r\n");
					break;
				}
				temp = GetToken(line,&indexb,":\r\n");
				strcat(middle,temp);
				LocalFree(temp);	
			
			} while (strchr(line+indexb,':'));
			// get rid of trailing space in middle
			middle[strlen(middle)-1]=0;
		}
		else 
			middle = GetToken(line,&indexb,"\r\n");
		
		
		out=Act(dest, source, command, middle, trailing);

		if (command)
			LocalFree(command);		

		if (middle)
			LocalFree(middle);		

		if (trailing)
			LocalFree(trailing);				

		LocalFree(source);
		return out;
		
		
	}
	
	/****************************************************************
	This function splits data read from the server and splits it into
	lines and passes it to a parsing procedure.
	*****************************************************************/
	
	char * ParseInString(char * buff, char *dest)
	{
		
		static char partial[1000];
		char * pCR, *out, *line;
		
		
		
		out=NULL;
		strcpy(dest,"status");
		
		if (buff!=NULL)
			strcat(partial,buff);
		
		if ((partial[0]!=0) && ((pCR=strchr(partial,'\n'))!=NULL))
		{
			line=(char *)LocalAlloc(LMEM_FIXED,1000);
			strncpy(line,partial,1+pCR-partial);
			line[1+pCR-partial]=0;
			
			if (*(pCR+1)==0)
				partial[0]=0;
			else
			{
				if ((pCR-partial) < 1000)
					memmove(partial,1+pCR,1000 - (pCR-partial));
				else
				{
					partial[0]=0;
				}
				
			}
			
			out=ParseIRC(line, dest);
			LocalFree(line);
		}
		
		return out;
		
	}
	
	/****************************************************************
	Returns TRUE the user is present in a channel already.
	FALSE otherwise.
	*****************************************************************/
	
	BOOL ChannelExists(char *channel)
	{
		
#ifdef _WIN32_WCE
		WCHAR wchannel[20];		
#endif
		
		
#ifdef _WIN32_WCE
		mbstowcs( wchannel, channel, 20 );
		if (SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)wchannel)==CB_ERR)
			return FALSE;
		else
			return TRUE;
#else
		if (SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)channel)==CB_ERR)
			return FALSE;
		else
			return TRUE;
#endif
		
		
	}
	
	
	/****************************************************************
	This function sets up a new channel based on a window type 
	and selects it as the current channel based on the select_channel 
	boolean.
	*****************************************************************/
	
	void SetupNewChannel(char *channel, BOOL select_channel, int window_type, int sock)
	{
		
#ifdef _WIN32_WCE
		WCHAR wchannel[100];		
#endif
	
		if (ChannelExists(channel))
			return;

#ifdef _WIN32_WCE
		mbstowcs( wchannel, channel, 100 );
		if (SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)wchannel)==CB_ERR)
		{
			SendMessage(hWndBox,CB_ADDSTRING,0,(LPARAM)(LPCTSTR)wchannel);
			CreateChannel(channel, window_type, sock);
		}
		if (select_channel)
		{
			SendMessage(hWndBox,CB_SELECTSTRING,-1,(LPARAM)(LPCTSTR)wchannel);
			MakeChannelCurrent(channel);
		}
#else
		if (SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)channel)==CB_ERR)
		{
			SendMessage(hWndBox,CB_ADDSTRING,0,(LPARAM)(LPCSTR)channel);
			CreateChannel(channel, window_type, sock);
		}
		if (select_channel)
		{
			SendMessage(hWndBox,CB_SELECTSTRING,-1,(LPARAM)(LPCSTR)channel);
			MakeChannelCurrent(channel);
		}
#endif
		
	}
	
	/****************************************************************
	Selects a string in the combobox that holds a list of all the 
	channels the user is present in.
	*****************************************************************/
	
	
	void SelectChannelInChannelBox(char *channel)
	{
		
		int idx;
#ifdef _WIN32_WCE
		WCHAR wchannel[100];
		
#endif
		
		
#ifdef _WIN32_WCE
		mbstowcs( wchannel, channel, 100 );
		if ((idx=SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)wchannel))!=CB_ERR)
			SendMessage(hWndBox,CB_SETCURSEL,(WPARAM)idx,(LPARAM)0);
		
#else
		if ((idx=SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)channel))!=CB_ERR)
			SendMessage(hWndBox,CB_SETCURSEL,(WPARAM)idx,(LPARAM)0);
#endif	
		
		
		
		
	}
	
	/****************************************************************
	Removes a string from the channel box.
	*****************************************************************/
	
	void RemoveChannelFromChannelBox(char *channel)
	{
		int idx;
#ifdef _WIN32_WCE
		WCHAR wchannel[100];
		
#endif
		
		
#ifdef _WIN32_WCE
		mbstowcs( wchannel, channel, 100 );
		if ((idx=SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)wchannel))!=CB_ERR)
			SendMessage(hWndBox,CB_DELETESTRING,(WPARAM)idx,(LPARAM)0);
#else
		if ((idx=SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)channel))!=CB_ERR)
			SendMessage(hWndBox,CB_DELETESTRING,(WPARAM)idx,(LPARAM)0);
#endif	
	}
	
	
	/****************************************************************
	This function produces an action based on a IRC "mode" change.
	*****************************************************************/
	
	void ActMode(char *channel, char *to, char *modes)
	{
		
		while (*modes!=0)
		{
			
			if (*modes=='+')
			{
				modes++;
				if (*modes=='o')
					AddMode(channel,to,'@');
				else if (*modes=='v')
					AddMode(channel,to,'+');
				
			}
			else if (*modes=='-')
			{
				modes++;
				if (*modes=='o')
					RemoveMode(channel,to,'@');
				else if (*modes=='v')
					RemoveMode(channel,to,'+');
			}
			
			modes++;
		}
		
		
		
		
	}
	
	/****************************************************************
	This function adds a "mode" to a user.
	*****************************************************************/
	
	void AddMode(char *channel,char *to,char mode)
	{
		
		char * nick;
		struct Names * name;
		char buff[20];
		char oldname[60];
		int indexb;
		struct Channel *chan;
		
		indexb = 0;
		chan = GetChannel(channel);
		while ((nick = GetToken(to,&indexb," ")) && nick[0]!=0)
		{
			indexb++;
			name = FindNameInChannel(chan, nick);
			if (name !=NULL)
			{
				if (strchr(name->modes,mode)==NULL)
				{
					buff[0]=mode;
					buff[1]=0;
					
					if (name->modes[0]!=0)
						strcat(&buff[1],name->modes);
					
					strcpy(oldname,name->modes);
					strcat(oldname,nick);
					
					strcpy(name->modes, buff);
					strcat(buff,nick);
					//		MessageBox(NULL, oldname, buff, MB_OK);
					ChangeNameInInfoWin(oldname, buff);
				}
			}
			LocalFree(nick);
		}
		
	}
	
	
	/****************************************************************
	This function removes a "mode" from a user.
	*****************************************************************/
	
	void RemoveMode(char *channel,char *to,char mode)
	{
		
		char * nick;
		struct Names * name;
		struct Channel *chan;
		char buff[20];
		char oldname[60];
		int i,ii;
		char c;
		int indexb;
		
		indexb = 0;
		
		chan = GetChannel(channel);
		while ((nick = GetToken(to,&indexb," ")) && nick[0]!=0)
		{
			indexb++;
			name = FindNameInChannel(chan, nick);
			if (name !=NULL)
			{
				i=0;
				ii=0;
				
				while ((c=name->modes[i++])!=0)
				{
					if (c!=mode)
					{
						buff[ii++]=c;
					}
				}
				
				buff[ii]=0;
				strcpy(oldname,name->modes);
				strcat(oldname,nick);
				
				strcpy(name->modes,buff);
				strcat(buff,nick);
				ChangeNameInInfoWin(oldname, buff);
			}
			LocalFree(nick);
		}
		
		
	}
	
	/****************************************************************
	Called when a user joins a channel.
	*****************************************************************/
	
	void ActJoin(char * nick, char *channel)
	{
		
		if (strcmpi(nick,Nickname)==0)
		{
			SetupNewChannel(channel, TRUE, WINDOW_TYPE_CHANNEL, csock);
			MakeChannelCurrent(channel);
		}
		else
			AddNameToChannel(channel,nick);
		
		
	}
	
	/****************************************************************
	Called when a user leaves a channel.
	*****************************************************************/
	
	void ActPart(char * nick, char *channel)
	{
#ifdef _WIN32_WCE
		WCHAR wchannel[100];
#endif
		
		int index;
		
		if (strcmpi(nick,Nickname)==0)
		{
			
			
#ifdef _WIN32_WCE
			mbstowcs( wchannel, channel, 100 );
			if ((index = SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)wchannel))!=CB_ERR)
			{
				SendMessage(hWndBox,CB_DELETESTRING,(WPARAM)index,(LPARAM)0);
				DeleteChannel(channel);
			}
#else
			if ((index = SendMessage(hWndBox,CB_FINDSTRINGEXACT,-1,(LPARAM)(LPCSTR)channel))!=CB_ERR)
			{
				SendMessage(hWndBox,CB_DELETESTRING,(WPARAM)index,(LPARAM)0);
				DeleteChannel(channel);
			}
#endif
			
			
		}
		
		RemoveNameFromChannel(channel,nick);
		
	}
	
	
	/****************************************************************
	Called when a user quits the server.
	*****************************************************************/
	
	void ActQuit(char * nick, char *trailing)
	{
		struct Names * pNamesList, *pPrevNamesList;
		struct Channel * chan;
		char fullname[40];
		char *out =(char *)LocalAlloc(LMEM_FIXED, 150);
		
		
		if (strcmpi(nick,Nickname)==0)
		{	
			InvalidateRect(hMainWnd,NULL,TRUE); 
			UpdateWindow(hMainWnd);
		}
		
		
		
		chan = pChannelList;
		
		while (chan)
		{
			pNamesList = chan->pNamesList;
			pPrevNamesList = NULL;
			
			while (pNamesList!=NULL && (strcmpi(pNamesList->name,nick)!=0))
			{
				pPrevNamesList = pNamesList;
				pNamesList = pNamesList->next;
			}
			
			if (pNamesList!=NULL)
			{
				
				if (pPrevNamesList == NULL)
					chan->pNamesList = pNamesList->next;
				else
					pPrevNamesList->next = pNamesList->next;
				
				
				
				
				if (strcmpi(chan->name, CurChannel)==0)
				{	
					
					strcpy(fullname,pNamesList->modes);
					strcat(fullname,nick);
					RemoveNameFromInfoWin(fullname);
				}
				strcpy(out,"*****");
				if (nick)
					strcat(out,nick);
				strcat(out," has quit");
				strcat(out," (");
				if (trailing)
					strcat(out,trailing);
				strcat(out,").*****\n");
				PrintToScreen(out, chan->name);
				
				LocalFree(pNamesList);
				
				
			}
			
			
			chan=chan->next;
		}
		
		
		LocalFree(out);
		
	}
	
	
	/****************************************************************
	Called when a user changes their nickname.
	*****************************************************************/
	
	void ActNick(char * nick, char *newname)
	{
		
		struct Names * pNamesList;
		struct Channel * chan;
		char fullname[40],fullnewname[40];
		
		char *out =(char *)LocalAlloc(LMEM_FIXED, 150);
		
		
		if (strcmpi(nick,Nickname)==0)
			strcpy(Nickname,newname);
		
		
		chan = pChannelList;
		
		while (chan)
			
		{
			pNamesList = chan->pNamesList;
			
			while (pNamesList!=NULL && (strcmpi(pNamesList->name,nick)!=0))
			{
				pNamesList = pNamesList->next;
			}
			
			if (pNamesList!=NULL)
			{
				strcpy(pNamesList->name, newname);
				
				
				if (strcmpi(chan->name, CurChannel)==0)
				{	
					strcpy(fullname,pNamesList->modes);
					strcat(fullname,nick);
					strcpy(fullnewname,pNamesList->modes);
					strcat(fullnewname,newname);
					
					ChangeNameInInfoWin(fullname, fullnewname);
				}
				
				strcpy(out,"*****");
				strcat(out,nick);
				strcat(out," is now known as ");
				strcat(out,newname);
				strcat(out,".*****\n");
				PrintToScreen(out, chan->name);
			}
			
			chan=chan->next;
		}
		
		LocalFree(out);
		
	}
	
	/****************************************************************
	Called when a user is kicked from a channel.
	*****************************************************************/
	
	void ActKick(char * nick, char *channel)
	{			
		ActPart(nick,channel);
	}
	
	/****************************************************************
	Returns a channel structure based on a textual name.
	*****************************************************************/
	
	struct Channel * GetChannel(char * chan)
	{
		
		struct Channel * retchan, *prevchan;
		
		
		retchan = pChannelList;
		prevchan = pChannelList;
		
		if (chan==NULL || *chan == 0)
			return retchan;
		
		while (retchan!=NULL && (strcmpi(retchan->name,chan)!=0))
		{
			prevchan = retchan;
			retchan=retchan->next;
		}
		
		if (retchan==NULL)
			retchan = prevchan;
		
		return retchan;
		
	}
	
	/****************************************************************
	Adds a nickname to a channel.
	*****************************************************************/
	
	void AddNameToChannel(char *channel, char *name)
	{
		
		struct Names * oldpNameList, *NewName;
		struct Channel * chan;
		
		chan = GetChannel(channel);
		
		if (FindNameInChannel(chan, name)!=NULL)
			return;

		oldpNameList = chan->pNamesList;
		NewName =(Names *)LocalAlloc(LMEM_FIXED, sizeof(Names));
		NewName->modes[0]=0;
		strcpy(NewName->name,name);
		NewName->next = oldpNameList;
		chan->pNamesList = NewName;
		
		if (strcmpi(channel, CurChannel)==0)
			AddNameToInfoWin(name);
		
	}
	
	/****************************************************************
	Add a list of names to the channel and strip off all the mode 
	info.
	*****************************************************************/
	
	
	void AddNamesToChannel(char *channel, char *names)
	{
		
		struct Names * oldpNameList, *NewName;
		struct Channel * chan;
		int indexb, i, names_length;
		char * aname;
		char c;
		char modes[4];
		
		chan = GetChannel(channel);

		if (strcmpi(chan->name, "status")==0)
			return;


		indexb=0;
		names_length = strlen(names);
		SendMessage(hWndInfoWin, WM_SETREDRAW ,FALSE, 0);
		while (indexb < names_length)
		{
			aname=GetToken(names,&indexb," \n");
			
			i=0;
			while ((c = aname[i])=='@' || (c=='+'))
			{
				modes[i++]=c;
			}
			
			modes[i]=0;
			indexb++;
			
			if (FindNameInChannel(chan, aname+i)==NULL)
			{
				oldpNameList = chan->pNamesList;
				NewName =(Names *)LocalAlloc(LMEM_FIXED, sizeof(Names));
				strcpy(NewName->name,aname + i);
				strcpy(NewName->modes,modes);
				NewName->next = oldpNameList;
				chan->pNamesList = NewName;
				
				if (strcmpi(channel, CurChannel)==0)
					AddNameToInfoWin(aname);
			}

			LocalFree(aname);
		}
		SendMessage(hWndInfoWin, WM_SETREDRAW ,TRUE, 0);
		InvalidateRect(hWndInfoWin , NULL, TRUE);
	}

	/****************************************************************
	Removes a name from all the channels a user is present on.
	*****************************************************************/
	
	
	void RemoveNamesFromChannel(struct Channel *chan)
	{
		struct Names * pNames;
		
		pNames = chan->pNamesList;
		
		while (pNames)
		{
			pNames=chan->pNamesList->next;
			LocalFree(chan->pNamesList);
			chan->pNamesList=pNames;
		}	
		SendMessage(hWndInfoWin ,LB_RESETCONTENT ,0 ,0);
	}
	
	/****************************************************************
	Removes a name from all the channels a user is present on.
	*****************************************************************/
	
	
	void RemoveNameFromAllChannels(char *name)
	{
		
		struct Names * pNamesList, *pPrevNamesList;
		struct Channel * chan;
		char fullname[40];
		
		chan = pChannelList;
		
		while (chan)
		{
			pNamesList = chan->pNamesList;
			pPrevNamesList = NULL;
			
			while (pNamesList!=NULL && (strcmpi(pNamesList->name,name)!=0))
			{
				pPrevNamesList = pNamesList;
				pNamesList = pNamesList->next;
			}
			
			if (pNamesList!=NULL)
			{
				
				if (pPrevNamesList == NULL)
					chan->pNamesList = pNamesList->next;
				else
					pPrevNamesList->next = pNamesList->next;
				
				
				
				
				if (strcmpi(chan->name, CurChannel)==0)
				{	
					
					strcpy(fullname,pNamesList->modes);
					strcat(fullname,name);
					RemoveNameFromInfoWin(fullname);
				}
				LocalFree(pNamesList);
				
			}
			
			
			chan=chan->next;
		}
		
	}
	
	/****************************************************************
	Removes a name from a single channel.
	*****************************************************************/
	
	void RemoveNameFromChannel(char *channel, char *name)
	{
		
		struct Names * pNamesList, *pPrevNamesList;
		struct Channel * chan;
		char fullname[40];
		
		chan = GetChannel(channel);
		pNamesList = chan->pNamesList;
		pPrevNamesList = NULL;
		
		while (pNamesList!=NULL && (strcmpi(pNamesList->name,name)!=0))
		{
			pPrevNamesList = pNamesList;
			pNamesList = pNamesList->next;
		}
		
		if (pNamesList!=NULL)
		{
			
			if (pPrevNamesList == NULL)
				chan->pNamesList = pNamesList->next;
			else
				pPrevNamesList->next = pNamesList->next;
			
			
			if (strcmpi(channel, CurChannel)==0)
			{
				strcpy(fullname,pNamesList->modes);
				strcat(fullname,name);
				RemoveNameFromInfoWin(fullname);
			}
			LocalFree(pNamesList);
			
		}
		
		
	}
	
	
	/****************************************************************
	Changes a nickname on a single channel.
	*****************************************************************/
	
	void ChangeNameInChannel(char *channel, char *name, char *newname, char *newmodes)
	{
		
		struct Names * pNamesList;
		struct Channel * chan;
		char fullname[40],fullnewname[40];
		
		chan = GetChannel(channel);
		pNamesList = chan->pNamesList;
		
		while (pNamesList!=NULL && (strcmpi(pNamesList->name,name)!=0))
		{
			pNamesList = pNamesList->next;
		}
		
		if (pNamesList!=NULL)
		{
			if (newmodes != NULL)
				strcpy(pNamesList->modes, newmodes);
			
			strcpy(pNamesList->name, newname);
			
			if (strcmpi(channel, CurChannel)==0)
			{
				strcpy(fullname,pNamesList->modes);
				strcat(fullname,name);
				strcpy(fullnewname,pNamesList->modes);
				strcat(fullnewname,newname);
				
				ChangeNameInInfoWin(fullname, fullnewname);
			}
			
			
		}
		
		
	}
	
	/****************************************************************
	Finds a nickname in a channel.
	*****************************************************************/
	
	struct Names * FindNameInChannel(struct Channel *chan, char *name)
	{
		struct Names * pNamesList;
		
		pNamesList = chan->pNamesList;
		
		while (pNamesList!=NULL && (strcmpi(pNamesList->name,name)!=0))
		{
			pNamesList = pNamesList->next;
		}
		
		
		return pNamesList;
	}
	
	
	/****************************************************************
	Makes a channel the current one.
	*****************************************************************/
	
	void MakeChannelCurrent( char * channel)
	{
		strcpy(CurChannel,channel);
		ClearNamesFromInfoWin();
		InvalidateRect(hMainWnd,NULL,TRUE); 
		UpdateWindow(hMainWnd);	
		SelectChannelInChannelBox(channel);
		FillInfoWin(CurChannel);
	}
	

	/*****************************************************************
	Strips modes from a nickname and passes it back
	*****************************************************************/
	char *GetNickname( char *nickname)
	{
		char *ptr;

		ptr=nickname;

		while((*ptr=='@' || *ptr=='+') && *ptr!=0)
			ptr++;

		return ptr;

	}